/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/*********************************************************************/
/* File seq_lex.c                                                    */
/* The lexicographic scanner for the sequence description parser     */
/* 1997 Robert Siegmund                                              */
/* fixed for 68000 compiler Dez 97 TW                                */
/*********************************************************************/

#include "xpciapi.h"
#include "xseqlex.h"
#include "xseqbool.h"
#include "xseqpars.h"
#include <xberror.h>
#include <string.h>
/*#include <fwutil.h> */

extern char yy_sequence_buffer[YYSEQBUFSIZE]; 
extern int  yy_seq_lval;
extern int  yybufptr;
extern char yyerrflag;
extern int  yychar;
extern int seq_tran_count;

/* the current sequencer descriptor which contains info */
/* about states, inputs and outputs of the sequencer    */

extern sequencer_descriptor * yy_seq__seq_descr;
extern bx_handletype BX_GLOBAL_handle;

#define SEQ_INPUT_TOKENS     (yy_seq__seq_descr -> input_tokens) + 2
#define SEQ_INPUT_VARS       (yy_seq__seq_descr -> inputs)
#define SEQ_OUTPUT_TOKENS    (yy_seq__seq_descr -> output_tokens)
#define SEQ_OUTPUT_VARS      (yy_seq__seq_descr -> outputs)

extern char yy01context;      /* shows context of symbol '0' and '1' */

extern bx_errtype yaccerr;

int yy_seq_error(s)
  char * s;
{
  s; /*error_msg(s);   */   
  yaccerr = BX_E_SEQ_SYNTAX_ERR;
  yyerrflag = 1;
  
  if (BX_GLOBAL_handle!=0xffffffff)
  {
    /*
      BX_GLOBAL_handle is set to handle before calling yy_seq_parse()
      and set to 0xffffffff afterwards. So BX_GLOBAL_handle 
      should have always a correct handle value here, as long
      as we do not work with multiple processes.
    */
    BestXLastErrorParamSet(BX_GLOBAL_handle,BX_ERRPAR_2,seq_tran_count);
  }
  else
  {
    /* Serious error: You should be aware that the CAPI is not
       reentrant. Surely you work with multiple processes here.
       Possible solution: put OS semaphore around the call to
       yy_seq_parse(), if you really need reentrancy here.
    */
  }
 
  return(0);
}

/***********************************************/
/* Lex scanner                                 */
/***********************************************/

int yy_seq_lex()
{
  int i = 0;
  int is_decimal = 1;
  char c;
  char token[16];

  do {
    /* in case the buffer is empty - return 0 */
       if(yybufptr == (int)BESTX_STRLEN(yy_sequence_buffer))
         return 0;

       c = yy_sequence_buffer[yybufptr++];

       if((c >= 0x30 && c <= 0x39) || (c >= 0x41 && c <= 0x5a) ||
          (c >= 0x61 && c <= 0x7a) || (c == '"') || (c == '_'))
          token[i++] = c;
       else
         {
           token[i] = '\0';

           /* check if token is a constant vector */

           if(token[0] == '"' && token[i-1] == '"')
             {
               yy_seq_lval = string_to_bin(token);
               yybufptr--;
               return(CONST_VECTOR);
             }

           /* check if token is a decimal number  */

           is_decimal = 1;

           for(i=0; i < (int)BESTX_STRLEN(token); i++)
             if(token[i] < 0x30 || token[i] > 0x39)
                is_decimal = 0;

           if(yy01context == 1 && (!BESTX_STRCMP(token,"0") || !BESTX_STRCMP(token,"1")))
                is_decimal = 0;

           if(is_decimal == 1 && BESTX_STRLEN(token))
              {
                yybufptr--;
                yy_seq_lval = string_to_dec(token);
                return(DECIMAL_NUM);
              }
           
           for(i = 0; i < SEQ_KEYWORDS && BESTX_STRLEN(token); i++)
             if(!BESTX_STRCMP(token,seq_keyword_list[i].name))
               {
                 yybufptr--;
                 return(seq_keyword_list[i].defined_as);
               }
             
           for(i = 0;i < SEQ_INPUT_TOKENS && BESTX_STRLEN(token); i++)
             if(!BESTX_STRCMP(token,yy_seq__seq_descr -> input_tokenlist[i].tok_name))
               {
                 yy_seq_lval = (yy_seq__seq_descr -> input_tokenlist[i]).position | (SEQ_INPUT_VARS << 8);
                 yybufptr--;   /* put back last character */

                 if((yy_seq__seq_descr -> input_tokenlist[i]).vectorlen == 1)
                     return(INPUT_VAR);
                 else
                     return(INPUT_VAR_VECTOR);
               }

            for(i = 0;i < SEQ_OUTPUT_TOKENS && BESTX_STRLEN(token); i++)
             if(!BESTX_STRCMP(token,(yy_seq__seq_descr -> output_tokenlist[i]).tok_name))
               {
                 yy_seq_lval = (yy_seq__seq_descr -> output_tokenlist[i]).position | (SEQ_OUTPUT_VARS << 8);
                 yybufptr--;   /* put back last character */
                  
                 if((yy_seq__seq_descr -> output_tokenlist[i]).vectorlen == 1)
                     return(TRANSIENT_OUTPUT);
                 else
                     return(TRANSIENT_OUTPUT_VECTOR);
               }
           
           if(BESTX_STRLEN(token)) 
               {
                 yybufptr--;
                 return(TRANSIENT_INSTANCE_NAME);
               }

           if(c == '<') yy01context = 1;     /* 0 and 1 to be interpreted as conditions */
           if(c == '[') yy01context = 0;
           if(c == ']') yy01context = 1;
           if(c == '>') yy01context = 0;     /* 0 and 1 to be interpreted as decimals   */

           switch(c) 
             {
              case '!':
               return (LOGICAL_NOT);
              case '&': /* catch double usage of &, as it is in the docu */
               if(yy_sequence_buffer[yybufptr] == '&') yybufptr++;
               return (LOGICAL_AND);
              case '|': /* catch double usage of | */
               if(yy_sequence_buffer[yybufptr] == '|') yybufptr++;
               return (LOGICAL_OR);
              case '^':
               return (LOGICAL_XOR);
              case '(':  /* everything else is caught as literal */
              case ')': 
              case ':':
              case '[':
              case ']':
              case '=':
              case '<':
              case '>': 
              case ';':
              case '.': return(c);
              case ' ':
              case '\n':
              case '\t':
              case '\r': continue;
              default : return(UNDEF_TOKEN);
             }
         }
     } while(c != '\0');

  return(0);

}


/***********************************************************************/
/* function string_to_bin()                                            */
/* converts the given binary number string to an integer               */
/***********************************************************************/

int string_to_bin(string)
   char * string;
{
   __int64 i;
   int j, result = 0;

   for(i = BESTX_STRLEN(string) - 2, j = 0; i > 0; i--, j++)
      {
        switch (string[i]) 
               {
                  case '0': break;
                  case '1': result += (1 << j); break; 
                  default: return (-1);
               }
      }
   return result;
}

/***********************************************************************/
/* function string_to_dec()                                            */
/* converts the given decimal number string to an integer              */
/***********************************************************************/

int string_to_dec(string)
   char * string;
{
   int i, j, result = 0;
   int power_of_10[9] = { 1, 10, 100, 1000, 10000, 100000, 1000000, \
                          10000000, 100000000 };

   for(i = 0, j = 0; i < (int)BESTX_STRLEN(string); i++, j++)
      {
        switch (string[BESTX_STRLEN(string) - i - 1]) 
               {
                  case '0': break;
                  case '1': result += 1 * power_of_10[j]; break;
                  case '2': result += 2 * power_of_10[j]; break;
                  case '3': result += 3 * power_of_10[j]; break;
                  case '4': result += 4 * power_of_10[j]; break;
                  case '5': result += 5 * power_of_10[j]; break;
                  case '6': result += 6 * power_of_10[j]; break;
                  case '7': result += 7 * power_of_10[j]; break;
                  case '8': result += 8 * power_of_10[j]; break;
                  case '9': result += 9 * power_of_10[j]; break;
                  default: return (-1);
               }
      }
   return result;
}
